#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.HL7AL.UnitTests
{
	/// <summary>
	/// Utility methods used in HL7AL unit tests.
	/// </summary>
	public class HL7ALUnitTestUtility : BaseTest 
	{
		// Message Date is now.
		private static readonly string _msgDateTime = System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":","");
		// DateTimeWanted is five days from now.
		private static readonly string _dateTimeWanted = System.DateTime.Now.AddDays(5).ToString("yyyyMMddhhmmsszzz").Replace(":","");
		// Lab Test Result date is today.
		private static readonly string _resultDate = System.DateTime.Now.ToString("yyyyMMdd");

		private static readonly string PATIENT_FILE_NAME_FIELD = ".01";
		private static readonly string PATIENT_FILE_SEX_FIELD = ".02";
		private static readonly string PATIENT_FILE_DOB_FIELD = ".03";
		private static readonly string PATIENT_FILE_SSN_FIELD = ".09";
		private static readonly string PATIENT_FILE_DATE_OF_DEATH_FIELD = ".351";
		private static readonly string PATIENT_FILE_ICN_FIELD = "991.01";
//		private static readonly string SELECT_PENDING_ORDER_TEST_STATEMENT = "SELECT TOP 1 OrderedTest.CprsOrderNumber, OrderableTest.CprsOrderableItemId FROM OrderedTest INNER JOIN OrderableTest ON OrderedTest.OrderableTestId = OrderableTest.OrderableTestId WHERE OrderStatusCode = 'P' AND CprsOrderNumber IS NOT NULL AND CprsOrderableItemId IS NOT NULL";
//		private static readonly string SELECT_PENDING_ORDER_COMPONENT_STATEMENT = "SELECT TOP 1 OrderedComponent.CprsOrderNumber, ComponentClass.CprsOrderableItemId FROM OrderedComponent INNER JOIN ComponentClass ON OrderedComponent.ComponentClassId = ComponentClass.ComponentClassId WHERE OrderStatusCode = 'P' AND CprsOrderNumber IS NOT NULL AND CprsOrderableItemId IS NOT NULL";
//		private static readonly string PLT_COMPONENT_ORDER = "4^PLATELETS^99VBC^^";
//		private static readonly string TAS_TEST_ORDER = "1^TAS^99VBC^^";
//		private static readonly string KC_DIVISION = "589^VA HEARTLAND - WEST, VISN 15";

		private HL7ALUnitTestUtility() {}

		/// <summary>
		/// Creates a CPRS new order message for unit testing only.
		/// </summary>
		/// <param name="test">The component or diagnostic test to be ordered.</param>
		/// <param name="divCode">Division Code</param>
		/// <returns>Valid HL7 CPRS order message.</returns>
		public static string GetCprsMessage(string orderControlCode, string orderNum, string groupNum, string test, string divCode, string vistaPatientId)
		{
			return BuildCprsMessage(orderControlCode, orderNum, groupNum, test, divCode, null, vistaPatientId);
		}

		/// <summary>
		/// Creates a Patient Merge message for unit testing only.
		/// </summary>
		/// <param name="fromPatientId">The VistaPatientId from VBECS representing the Merge From patient.</param>
		/// <param name="toPatientId">The VistaPatientId from VBECS representing the Merge To patient.</param>
		/// <returns>Valid HL7 Merge Message.</returns>
		public static string GetPatientMergeMessage(string fromPatientId, string toPatientId)
		{
			return BuildPatientMergeMessage(fromPatientId, toPatientId);
		}

		/// <summary>
		/// Creates a Patient Update message for unit testing only.
		/// </summary>
		/// <param name="patientId">The VistaPatientId of the patient being updated.</param>
		/// <param name="fieldChanged">The field number being updated. One of the static readonly constants.</param>
		/// <param name="changedValue">The value changed.</param>
		/// <returns>Valid HL7 Patient Update message</returns>
		public static string GetPatientUpdateMessage(string patientId, string fieldChanged, string changedValue)
		{
			return BuildPatientUpdateMessage(patientId, fieldChanged, changedValue);
		}


		private static string BuildCprsMessage(string orderControlCode, string orderNum,
			string groupNum, string test, string divCode, string vbecsOrderNum, string vistaPatientId )
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH|^~\\&|OERR|589|VBECS|589|");
			sb.Append(_msgDateTime);
			sb.Append("||OMG^O19|589");
			sb.Append(GetRandomNumber(999999));
			sb.Append("|T|2.4||||AL|US");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID|||^^^^NI~");
			sb.Append(GetRandomNumber(999999999));
			sb.Append("^^^^SS~");
			sb.Append(vistaPatientId);
			sb.Append("^^^^PI||");
			sb.Append("Unittest^Patient");
			sb.Append(GetRandomNumber(999));
			sb.Append("^X||");
			// Patient date of birth
			DateTime datePatientDob = DateTime.Now - System.TimeSpan.FromDays(10000);
			string _patientDob = datePatientDob.ToString("yyyyMMdd").Replace(":","");
			sb.Append( _patientDob + "|M");
			sb.Append(CARRIAGE_RETURN);
			// PV1 segment
			sb.Append("PV1||O|UNIT TEST PATIENT LOCATION|||||||");
			sb.Append(CARRIAGE_RETURN);
			// ORC segment
			sb.Append("ORC|");
			sb.Append(orderControlCode);
			sb.Append("|" + orderNum + "^OR|" + vbecsOrderNum + "^VBEC|" + groupNum + "^OR|||||");
			sb.Append(_dateTimeWanted);
			sb.Append("|53299^TOMLIN^BRIAN||53299^TOMLIN^BRIAN||||");
			if( orderControlCode == "CA" )
			{
				sb.Append("^Order canceled by unit test.|");
			}
			else
			{
				sb.Append("^Reason is unit test.|");
			}
			sb.Append(divCode);
			sb.Append("|||0^NO");
			sb.Append(CARRIAGE_RETURN);
			// OBR segment
			sb.Append("OBR|1|" + orderNum + "^OR||");
			sb.Append(test);
			sb.Append("|||||||L||12231;3041004;2;6479||^^||||||||||||2^^^^");
			sb.Append(_dateTimeWanted);
			sb.Append("^P^IMMEDIATE^Transurethral Resection (TURP)");
			sb.Append(CARRIAGE_RETURN);
			// NTE segment
			sb.Append("NTE|1||Unit Test Comments");
			sb.Append(CARRIAGE_RETURN);
			// DG1 segment
			sb.Append("DG1|1||^^^^^|||A");
			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		private static string BuildLabPreviousResultsMessage(string orderControlCode, string orderNum,
			string groupNum, string test, string divCode, string vbecsOrderNum, string vistaPatientId )
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH|^~\\&|OERR|589|VBECS|589|");
			sb.Append(_msgDateTime);
			sb.Append("||OMG^O19|589");
			sb.Append(GetRandomNumber(999999));
			sb.Append("|T|2.4||||AL|US");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID|||^^^^NI~");
			sb.Append(GetRandomNumber(999999999));
			sb.Append("^^^^SS~");
			sb.Append(vistaPatientId);
			sb.Append("^^^^PI||");
			sb.Append("Unittest^Patient");
			sb.Append(GetRandomNumber(999));
			sb.Append("^X||19440603|M");
			sb.Append(CARRIAGE_RETURN);
			// PV1 segment
			sb.Append("PV1||O|UNIT TEST PATIENT LOCATION|||||||");
			sb.Append(CARRIAGE_RETURN);
			// ORC segment
			sb.Append("ORC|");
			sb.Append(orderControlCode);
			sb.Append("|" + orderNum + "^OR|" + vbecsOrderNum + "^VBEC|" + groupNum + "^OR|||||");
			sb.Append(_dateTimeWanted);
			sb.Append("|53299^TOMLIN^BRIAN||53299^TOMLIN^BRIAN||||");
			if( orderControlCode == "CA" )
			{
				sb.Append("^Order canceled by unit test.|");
			}
			else
			{
				sb.Append("^Reason is unit test.|");
			}
			sb.Append(divCode);
			sb.Append("|||0^NO");
			sb.Append(CARRIAGE_RETURN);
			// OBR segment
			sb.Append("OBR|1|" + orderNum + "^OR||");
			sb.Append(test);
			sb.Append("|||||||L||12231;3041004;2;6479||^^||||||||||||2^^^^");
			sb.Append(_dateTimeWanted);
			sb.Append("^P^IMMEDIATE^Transurethral Resection (TURP)");
			sb.Append(CARRIAGE_RETURN);
			// OBX segment
			//			sb.Append("OBX|1||3^HGB^99LRT||13||13.0-18.0||||F|||" + _resultDate);
			sb.Append("OBX|1||3^HGB^99LRT||13||13.0-18.0||||F|||20001011175388-0600");

			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		private static string BuildPatientMergeMessage(string fromPatientId, string toPatientId)
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH^~|\\&^MPIF TRIGGER^^VBECS ADT^589^");
			sb.Append(_msgDateTime);
			sb.Append("^^ADT~A40^");
			sb.Append(GetRandomNumber(999999));
			sb.Append("^T^2.4^^^AL^AL^");
			sb.Append(CARRIAGE_RETURN);
			// EVN segment
			sb.Append("EVN^^^^^53296~TOMLIN~BRIAN~~~~~~USVHA&&0363~L~~~NI~VA FACILITY ID&589&L^^589");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID^1^V^");
			sb.Append(fromPatientId);
			sb.Append("~~~USVHA&&0363~PI~VA FACILITY ID&589&L^");
			sb.Append(CARRIAGE_RETURN);
			// MGR segment
			sb.Append("MRG^");
			sb.Append(toPatientId);
			sb.Append("~~~USVHA&&0363~PI~VA FACILITY ID&589&L^");
			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		private static string BuildPatientUpdateMessage(string patientId, string field, string changedValue)
		{
			StringBuilder sb = new StringBuilder();
			// MSH segment
			sb.Append("MSH^~|\\&^VAFC PIMS^589^VBECS ADT^589^");
			sb.Append(_msgDateTime);
			sb.Append("^^ADT~A08^");
			sb.Append(GetRandomNumber(99999999));
			sb.Append("^T^2.3^^^NE^NE^USA");
			sb.Append(CARRIAGE_RETURN);
			// EVN segment
			sb.Append("EVN^A08^20041007^^^TOMLIN,BRIAN N");
			sb.Append(CARRIAGE_RETURN);
			// PID segment
			sb.Append("PID^1^");
			if( field == PATIENT_FILE_ICN_FIELD )
			{
				sb.Append( changedValue );
			}
			else
			{
				sb.Append("");
			}
			sb.Append("^" + patientId + "~5~M10^2277^");
			if( field == PATIENT_FILE_NAME_FIELD )
			{
				sb.Append( changedValue );
			}
			else
			{
				sb.Append("UnitTest~Patient");
			}
			sb.Append("^^");
			if( field == PATIENT_FILE_DOB_FIELD )
			{
				sb.Append( changedValue );
			}
			else
			{
				sb.Append("19490120");
			}
			sb.Append("^");
			if( field == PATIENT_FILE_SEX_FIELD )
			{
				sb.Append( changedValue );
			}
			else
			{
				sb.Append("");
			}
			sb.Append("^^~~0005~~~CDC^1103 ASPEN~~ACRETON~KS~12345~~P~~077|~~~~~~VACAE~~~~~&|~~~~~~VACAA~~~~~&|~");
			sb.Append("~~~~~VACAC~~~~~&|~~~~~~VACAM~~~~~&|~~~~~~VACAO~~~~~&^^^^^^^^");
			if( field == PATIENT_FILE_SSN_FIELD )
			{
				sb.Append( changedValue );
			}
			else
			{
				sb.Append("");
			}
			//			sb.Append("^^^~~0189~~~CDC^^^^^^");
			sb.Append("^^^~~0189~~~CDC^");

			if( field == PATIENT_FILE_DATE_OF_DEATH_FIELD )
			{
				sb.Append( "PULLOVER VERMONT^^^^^^" + changedValue + "^Y");
			}
			else
			{
				sb.Append("");
			}
			sb.Append(CARRIAGE_RETURN);
			// PD1 segment
			sb.Append("PD1^^^VA HEARTLAND - WEST, VISN 15~~589^");
			sb.Append(CARRIAGE_RETURN);
			// PV1 segment
			sb.Append("PV1^1^O^^^^^^^^^^^^^^^^NON-VETERAN (OTHER)^^^^^^^^^^^^^^^^^^^^^^^^^^20041007^^^^^^1307096");
			sb.Append(CARRIAGE_RETURN);
			// OBX segment
			sb.Append("OBX^1");
			sb.Append(CARRIAGE_RETURN);
			// ZPD segment
			sb.Append("ZPD^1^COLLATERAL OF JAMES LYLE^^^^^^^^^0^^^^29^0^^0^^^");
			sb.Append(CARRIAGE_RETURN);
			// ZSP segment
			sb.Append("ZSP^1^0^^T^^^^");
			sb.Append(CARRIAGE_RETURN);
			// ZEL segment
			sb.Append("ZEL^1^13^888-33-2277~~^2277^^^^0^NON-VETERAN (OTHER)^^^^^^^^^^^^^");
			sb.Append(CARRIAGE_RETURN);
			// ZCT segment
			sb.Append("ZCT^1^1^^^^^^^");
			sb.Append(CARRIAGE_RETURN);
			// ZEM segment
			sb.Append("ZEM^1^1^^^^^^");
			sb.Append(CARRIAGE_RETURN);
			// ZFF segment
			sb.Append("ZFF^2^" + field + ";");
			sb.Append(CARRIAGE_RETURN);
			// ZIR segment
			sb.Append("ZIR^");
			sb.Append(CARRIAGE_RETURN);
			// ZEN segment
			sb.Append("ZEN^1");
			sb.Append(CARRIAGE_RETURN);

			return sb.ToString();

		}

		/// <summary>
		/// Static method used throughout the HL7 Unit Test project to generate random numbers.
		/// </summary>
		/// <param name="maxValue">The maximun value for the random number</param>
		/// <returns>A random number up to the maxValue.  Defaults to a maxValue of 999.</returns>
		public static int GetRandomNumber( int maxValue )
		{
			if( maxValue < 0 )
			{
				maxValue = 999;
			}
			return new Random().Next(maxValue);
		}

	}
}
#endif
